/*
 * Decompiled with CFR 0.152.
 */
package at.jku.ssw.mevss.cerberus.ci.interfaces.filetransfer;

import at.jku.ssw.mevss.cerberus.ci.interfaces.filetransfer.FileHeader;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.Iterator;
import java.util.logging.Logger;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

public class FileTransferProtocol {
    private final Logger logger;
    private final byte[] buffer = new byte[16384];

    public FileTransferProtocol(String name) {
        this.logger = Logger.getLogger("cerberus.filetransfer.protocol." + name);
    }

    public void send(String prefix, File file, ObjectOutputStream out) throws IOException {
        this.send(prefix, file, null, null, out);
    }

    public void send(String prefix, File file, String hiddenFilePrefix, String[] extensions, ObjectOutputStream out) throws IOException {
        if (file.exists()) {
            if (file.isFile() && ZipAccess.check(file)) {
                this.logger.info("sending " + file + " (reading as zip)");
                try (ZipAccess access = new ZipAccess(prefix, file);){
                    this.send(access, new ZipEntry("/"), hiddenFilePrefix, extensions, out);
                }
            }
            this.logger.info("sending " + file + " (reading as regular file)");
            try (FileAccess access = new FileAccess(prefix, file);){
                this.send(access, file.toPath(), hiddenFilePrefix, extensions, out);
            }
        }
        out.writeObject(null);
    }

    private <F> void send(Access<F> access, F file, String hiddenFilePrefix, String[] extensions, ObjectOutputStream out) throws IOException {
        if (Thread.currentThread().isInterrupted()) {
            throw new InterruptedIOException();
        }
        if (hiddenFilePrefix != null && access.getName(file).startsWith(hiddenFilePrefix)) {
            this.logger.info("skipping " + access.getName(file) + " because it is a hidden file");
            return;
        }
        if (!access.isDirectory(file) && extensions != null && extensions.length > 0 && !Arrays.stream(extensions).anyMatch(extension -> access.getName(file).endsWith((String)extension))) {
            this.logger.info("skipping " + access.getName(file) + " because it is not in the included extensions");
            return;
        }
        if (access.isLink(file)) {
            this.sendLink(access, file, out);
        } else if (access.isDirectory(file)) {
            this.sendDirectory(access, file, hiddenFilePrefix, extensions, out);
        } else if (access.isFile(file)) {
            this.sendFile(access, file, out);
        } else {
            this.sendFile(access, file, out);
        }
    }

    public void read(String prefix, File directory, ObjectInputStream in) throws ClassNotFoundException, IOException {
        FileHeader header = (FileHeader)((Object)in.readObject());
        while (header != null) {
            switch (header) {
                case DIRECTORY: {
                    this.readDirectory(prefix, directory, in);
                    break;
                }
                case FILE: {
                    this.readFile(prefix, directory, in);
                    break;
                }
                case LINK: {
                    this.readLink(prefix, directory, in);
                    break;
                }
                default: {
                    assert (false);
                    break;
                }
            }
            if (Thread.currentThread().isInterrupted()) {
                throw new InterruptedIOException();
            }
            header = (FileHeader)((Object)in.readObject());
        }
    }

    private <F> void sendDirectory(Access<F> access, F file, String hiddenFilePrefix, String[] extensions, ObjectOutputStream out) throws IOException {
        String name = access.getPathAndName(file);
        this.logger.info("sending directory " + file + " -> " + name);
        out.writeObject((Object)FileHeader.DIRECTORY);
        out.writeUTF(name);
        try (CloseableIterable<F> iterable = access.getChildren(file);){
            Iterator<F> iterator = iterable.iterator();
            while (iterator.hasNext()) {
                this.send(access, iterator.next(), hiddenFilePrefix, extensions, out);
            }
        }
    }

    private void readDirectory(String prefix, File directory, ObjectInputStream in) throws ClassNotFoundException, IOException {
        String name = in.readUTF();
        File file = new File((prefix != null ? prefix + File.separator : "") + directory + File.separator + name);
        this.logger.info("reading directory " + name + " -> " + file);
        file.mkdirs();
    }

    private <F> void sendFile(Access<F> access, F file, ObjectOutputStream out) throws IOException, FileNotFoundException {
        String name = access.getPathAndName(file);
        this.logger.info("sending file " + file + " -> " + name);
        out.writeObject((Object)FileHeader.FILE);
        out.writeUTF(name);
        out.writeBoolean(access.isExecutable(file));
        try (InputStream in = access.open(file);){
            int read = in.read(this.buffer);
            while (read > 0) {
                out.writeInt(read);
                out.write(this.buffer, 0, read);
                out.flush();
                read = in.read(this.buffer);
            }
            out.writeInt(0);
        }
    }

    private void readFile(String prefix, File directory, ObjectInputStream in) throws ClassNotFoundException, IOException {
        String name = in.readUTF();
        boolean executable = in.readBoolean();
        File file = new File((prefix != null ? prefix + File.separator : "") + directory + File.separator + name);
        this.logger.info("reading file " + name + " -> " + file);
        try (OutputStream out = this.createOutputStream(file);){
            int length = in.readInt();
            while (length > 0) {
                while (length > 0) {
                    int read = in.read(this.buffer, 0, Math.min(this.buffer.length, length));
                    if (read < 0) {
                        throw new IOException("Stream terminated unexpectedly");
                    }
                    out.write(this.buffer, 0, read);
                    length -= read;
                }
                length = in.readInt();
            }
        }
        file.setExecutable(executable);
    }

    protected InputStream createInputStream(File file) {
        try {
            FileInputStream in = new FileInputStream(file);
            file.setReadable(true);
            return in;
        }
        catch (FileNotFoundException e) {
            assert (false) : "here be dragons";
            return null;
        }
    }

    protected OutputStream createOutputStream(File file) {
        try {
            FileOutputStream out = new FileOutputStream(file);
            file.setReadable(true);
            return out;
        }
        catch (FileNotFoundException e) {
            assert (false) : "here be dragons";
            return null;
        }
    }

    private <F> void sendLink(Access<F> access, F file, ObjectOutputStream out) throws IOException {
        F dest = access.getDestination(file);
        String fileName = access.getPathAndName(file);
        String destName = dest.toString();
        this.logger.info("sending link " + file + " -> " + fileName + " (-> " + destName + ")");
        out.writeObject((Object)FileHeader.LINK);
        out.writeUTF(fileName);
        out.writeUTF(destName);
        out.writeBoolean(access.isExecutable(file));
    }

    private void readLink(String prefix, File directory, ObjectInputStream in) throws ClassNotFoundException, IOException {
        String srcName = in.readUTF();
        String destName = in.readUTF();
        File src = new File((prefix != null ? prefix + File.separator : "") + directory + File.separator + srcName);
        File dest = new File(destName);
        boolean executable = in.readBoolean();
        this.logger.info("reading link " + srcName + " -> " + src + " (-> " + dest + ")");
        if (src.exists()) {
            this.logger.info("link src " + src + " already exists, will be replaced");
            Files.delete(src.toPath());
        }
        Files.createSymbolicLink(src.toPath(), dest.toPath(), new FileAttribute[0]);
        src.setExecutable(executable);
    }

    private static final class ZipAccess
    extends Access<ZipEntry> {
        private static final String SEPARATOR = "/";
        private final File zip;
        private final ZipInputStream in;

        public ZipAccess(String prefix, File zip) throws IOException {
            super(prefix);
            this.zip = zip;
            this.in = new ZipInputStream(new FileInputStream(zip));
        }

        @Override
        public String getPathAndName(ZipEntry f) {
            return (this.prefix != null ? this.prefix + File.separator : "") + f.getName();
        }

        @Override
        public String getName(ZipEntry f) {
            return f.getName().substring(f.getName().indexOf(SEPARATOR) + 1);
        }

        @Override
        public boolean isDirectory(ZipEntry f) {
            return f.isDirectory();
        }

        @Override
        public boolean isLink(ZipEntry f) {
            return f.getComment() != null && f.getComment().startsWith("->");
        }

        @Override
        public boolean isFile(ZipEntry f) {
            return !f.isDirectory();
        }

        @Override
        public CloseableIterable<ZipEntry> getChildren(ZipEntry f) {
            if (f.getName().equals(SEPARATOR)) {
                return new CloseableIterable<ZipEntry>(){

                    @Override
                    public Iterator<ZipEntry> iterator() {
                        return new Iterator<ZipEntry>(){
                            private ZipEntry next = null;

                            @Override
                            public boolean hasNext() {
                                if (this.next != null) {
                                    return true;
                                }
                                try {
                                    this.next = in.getNextEntry();
                                    return this.next != null;
                                }
                                catch (IOException e) {
                                    return false;
                                }
                            }

                            @Override
                            public ZipEntry next() {
                                if (this.next != null) {
                                    ZipEntry result = this.next;
                                    this.next = null;
                                    return result;
                                }
                                if (this.hasNext()) {
                                    return this.next();
                                }
                                return null;
                            }
                        };
                    }

                    @Override
                    public void close() throws IOException {
                    }
                };
            }
            return new EmptyCloseableIterable<ZipEntry>();
        }

        @Override
        public boolean isExecutable(ZipEntry f) {
            return false;
        }

        @Override
        public InputStream open(ZipEntry f) throws IOException {
            ZipInputStream in = new ZipInputStream(new FileInputStream(this.zip));
            ZipEntry entry = in.getNextEntry();
            while (entry != null) {
                if (f.getName().equals(entry.getName())) {
                    return in;
                }
                entry = in.getNextEntry();
            }
            in.close();
            throw new FileNotFoundException(f.getName());
        }

        @Override
        public ZipEntry getDestination(ZipEntry f) throws IOException {
            return new ZipEntry(f.getComment().substring("->".length()));
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        public static boolean check(File file) {
            try (ZipInputStream in = new ZipInputStream(new FileInputStream(file));){
                ZipEntry entry = in.getNextEntry();
                boolean bl = entry != null;
                return bl;
            }
            catch (IOException __) {
                return false;
            }
        }

        @Override
        public void close() throws IOException {
            this.in.close();
        }
    }

    private class FileAccess
    extends Access<Path> {
        private final Path base;

        public FileAccess(String prefix, File base) {
            this(prefix, base.toPath());
        }

        public FileAccess(String prefix, Path base) {
            super(prefix);
            this.base = base;
        }

        @Override
        public String getPathAndName(Path f) {
            return (this.prefix != null ? this.prefix + File.separator : "") + this.base.toFile().toURI().relativize(f.toFile().toURI()).getPath();
        }

        @Override
        public String getName(Path f) {
            return f.toFile().getName();
        }

        @Override
        public boolean isDirectory(Path f) {
            return f.toFile().isDirectory();
        }

        @Override
        public boolean isLink(Path f) {
            return Files.isSymbolicLink(f);
        }

        @Override
        public boolean isFile(Path f) {
            return f.toFile().isFile();
        }

        @Override
        public CloseableIterable<Path> getChildren(Path f) {
            try {
                final DirectoryStream<Path> stream = Files.newDirectoryStream(f);
                return new CloseableIterable<Path>(){

                    @Override
                    public Iterator<Path> iterator() {
                        return stream.iterator();
                    }

                    @Override
                    public void close() throws IOException {
                        stream.close();
                    }
                };
            }
            catch (IOException e) {
                return new EmptyCloseableIterable<Path>(e);
            }
        }

        @Override
        public boolean isExecutable(Path f) {
            return Files.isExecutable(f);
        }

        @Override
        public InputStream open(Path f) throws FileNotFoundException {
            return FileTransferProtocol.this.createInputStream(f.toFile());
        }

        @Override
        public Path getDestination(Path f) throws IOException {
            return Files.readSymbolicLink(f);
        }

        @Override
        public void close() {
        }
    }

    private static final class EmptyCloseableIterable<F>
    implements CloseableIterable<F> {
        private final IOException ioe;

        public EmptyCloseableIterable() {
            this(null);
        }

        public EmptyCloseableIterable(IOException ioe) {
            this.ioe = ioe;
        }

        @Override
        public Iterator<F> iterator() {
            return new Iterator<F>(){

                @Override
                public boolean hasNext() {
                    return false;
                }

                @Override
                public F next() {
                    return null;
                }
            };
        }

        @Override
        public void close() throws IOException {
            if (this.ioe != null) {
                throw new IOException(this.ioe);
            }
        }
    }

    private static interface CloseableIterable<F>
    extends Closeable {
        public Iterator<F> iterator();
    }

    private static abstract class Access<F>
    implements Closeable {
        protected final String prefix;

        protected Access(String prefix) {
            this.prefix = prefix;
        }

        public abstract String getPathAndName(F var1);

        public abstract String getName(F var1);

        public abstract boolean isDirectory(F var1);

        public abstract boolean isLink(F var1);

        public abstract boolean isFile(F var1);

        public abstract CloseableIterable<F> getChildren(F var1);

        public abstract boolean isExecutable(F var1);

        public abstract InputStream open(F var1) throws IOException;

        public abstract F getDestination(F var1) throws IOException;
    }
}

