package spec.benchmarks._225_shock;
//-------------------------begin header-------------------------------------
//------------------------------------------------------------------------
// BoundaryThread - this object embodies the computation kernel for
//  		flux across two finite difference cells in the Godunov
//		method.  Includes iterative Riemann solver.  Requires
//		only the left and right interface conditions plus any 
//		2nd order vanLeer terms to determine net flux across interface.
//
// (c) 1997.
//
// Written originally in FORTRAN and ported to Java by:
// James C. Liu, X.M. Chen, PF Peterson, P. Collela, V.E. Schrock
// Dept. of Nuclear Engineering and Dept. of Mech. Engineering, 
// University of California at Berkeley.
//
// Adopted from TSUNAMI (Transient Shockwave Upwind Numerical Analysis Method
// for ICF) code courtesy of James Liu, and UC Berkeley Department of Nuclear
// Engineering.
//
// Algorithms based on CFD methods published by Colella, et al., Dept.
// of Mechanical Engineering, UC Berkeley.
//
// Research Performed by UCB Dept. of Nuclear Engineering are under the
// auspices of the US Dept. of Energy for the Lawrence Livermore National
// Laboratory under contract W-7405-ENG-48.
//------------------------------------------------------------------------
// Permission to use, modify and distribute this source is hereby granted for
// educational and benchmarking purposes provided that this header is included
// in subsequent modifications.
//
// Neither the author(s) nor the University of California and LLNL/DOE make
// any claims of computational accuracy or fitfullness for engineering uses
// or take any responsibility for any damages to users incurred directly or
// indirectly by the usage of this source.  Users accept all the consequences.
//------------------------------------------------------------------------
//-------------------------end header-------------------------------------


public class BoundaryThread extends Thread {

    private ComputeTracker tracker = null;
    private double massflux, momentumflux,energyflux;
    private double ql[],qr[],qtl[],qtr[],ql0[],qr0[],qleft[],qright[];
    private double dql[],dqr[];
    private double cl,cr,dt,dx,xbyt;
    private double gamma, gm1, gp1, zero, half, one, two;

    public BoundaryThread(ComputeTracker tracker) {
	this.tracker = tracker;
	massflux = 0.0;
	momentumflux = 0.0;
	energyflux = 0.0;
	xbyt = 0.0;
	gamma = 1.4;
	gm1 = gamma - 1.0;
	gp1 = gamma + 1.0;
	zero = 0.0;
	half = 0.5;
	one = 1.0;
	two = 2.0;

	ql = new double[3];
	qr = new double[3];
	qtl = new double[3];
	qtr = new double[3];
	ql0 = new double[3];
	qr0 = new double[3];
	qleft = new double[3];
	qright = new double[3];
	dql = new double[3];
	dqr = new double[3];
    }

    public void run() {
	computeLeftRight();
	computeRiemann();
	tracker.completedCell(1);
    }

    public void setLRConditions(double rhol, double rhor, double ul, double ur,
	double pl, double pr, double cl, double cr, double drhol, double drhor,
	double dul, double dur, double dpl, double dpr, double dt, double dx,
	double gamma) {

	this.ql[0] = rhol;
	this.qr[0] = rhor;
	this.ql[1] = ul;
	this.qr[1] = ur;
	this.ql[2] = pl;
	this.qr[2] = pr;
	this.cl = cl;
	this.cr = cr;
	this.dql[0] = drhol;
	this.dqr[0] = drhor;
	this.dql[1] = dul;
	this.dqr[1] = dur;
	this.dql[2] = dpl;
	this.dqr[2] = dpr;
	this.dt = dt;
	this.dx = dx;
	this.gamma = gamma;
	gm1 = gamma - 1.0;
	gp1 = gamma + 1.0;
    }

    protected void computeLeftRight() {
	double charlm, charlp, charrm, charrp,psml,psmr;
	double betalm, betal0, betarp, betar0;
	double dtdx = dt/dx;
	int i;

	//---left/right, plus/minus characteristics
	charlm = ql[1] - cl;
	charlp = ql[1] + cl;
	charrm = qr[1] - cr;
	charrp = qr[1] + cr;

	//---left states to send to riemann solver
	for(i=0;i<3;i++) {
	    qtl[i] = ql[i]+half*(one-dtdx*Math.max(charlp,zero))*dql[i];
	    qtr[i] = qr[i]-half*(one+dtdx*Math.min(charrm,zero))*dqr[i];
	    ql0[i] = ql[i]+half*(one-dtdx*ql[1])*dql[i];
	    qr0[i] = ql[i]-half*(one+dtdx*qr[1])*dqr[i];
	}
	betalm = -(dql[1]-dql[2]/ql[0]/cl)*dtdx*half/ql[0];
	betal0 = -(dql[2]/Math.pow((ql[0]*cl),two) - dql[0]/ql0[0]/qtl[0])*cl*dtdx*half;
	betarp = -(dqr[1]+dqr[2]/qr[0]/cr)*dtdx*half/qr[0];
	betar0 = (dqr[2]/Math.pow((qr[0]*cr),two) - dqr[0]/qr0[0]/qtr[0])*cr*dtdx*half;

	betalm = (charlm > zero) ? betalm : zero;
	betal0 = (ql[1] > zero) ? betal0 : zero;
	betarp = (charrp < zero) ? betarp: zero;
	betar0 = (qr[1] < zero) ? betarp: zero;

	qleft[0] = one/(one/qtl[0] - betal0 - betalm);
	qleft[1] = qtl[1] - betalm*cl*ql[0];
	qleft[2] = qtl[2] - betalm*Math.pow((cl*ql[0]),two);
	qright[0] = one/(one/qtr[0] - betar0 - betarp);
	qright[1] = qtr[1] + betarp*cr*qr[0];
	qright[2] = qtl[2] + betarp*Math.pow((cr*qr[0]),two);

	//----set density and pressure floors to speed convergence
	qleft[0] = Math.max(1.0e-3,qleft[0]);
	qright[0] = Math.max(1.0e-3,qright[0]);
	psml = Math.max(1.0e-6,(half*1.0e-06*ql[0]*Math.pow(ql[1],two)));
	psmr = Math.max(1.0e-6,(half*1.0e-06*qr[0]*Math.pow(qr[1],two)));
	qleft[2] = Math.max(psml,qleft[2]);
	qright[2] = Math.max(psmr,qright[2]);
    }

    protected void computeRiemann() {
	double p,u,r;
	double pold,csl,csr, al,ar,psml,psmr;
	double rl = qleft[0];
	double rr = qright[0];
	double ul = qleft[1];
	double ur = qright[1];
	double pl = qleft[2];
	double pr = qright[2];
	int ncounter, i;

	//---set iterations here
	int niter = 5;

	//----quick check to see if this is a contact surface, if so, 
	//----we don't have to iterate...just advect quantities
	if(ul==ur && pl==pr) {
	    r = (ul>zero) ? rl:rr;
	    u = ul;
	    p = pl;
	    computeFlux(r,u,p);
	    return;
	}

	psml = Math.max(1.0e-6,(half*1.0e-6*rl*ul*ul));
	psmr = Math.max(1.0e-6,(half*1.0e-6*rr*ur*ur));
	csl = Math.sqrt(gamma*pl/rl);
	csr = Math.sqrt(gamma*pr/rr);
	al = pl/Math.pow(rl,gamma);
	ar = pr/Math.pow(rr,gamma);
	pold = half*(pl+pr);
	
	p = (rr*csr*pl+rl*csl*pr+rl*csl*rr*csr*(ul-ur))/(rr*csr+rl*csl);
	p = Math.max(p,(Math.max(psml,psmr)));
	u = zero;
	r = one;

	double pstar,ustar,ulss,ulsr,urss,ursr,ulstar,urstar;
	double taulss,taulsr,taurss,taursr,taulstar,taurstar;
	double clstar,crstar,zlr,zls,zrr,zrs,zl,zr,y,z;

	ulstar = urstar = clstar = crstar = zero;
	taulstar = taurstar = one;

	ncounter = 0;
	while (ncounter < niter) {

	    //----compute speed on left and right of contact surface-------
	    ulss=ul-(p-pl)/ws(p,pl,one/rl);
	    ulsr=ul+two/gm1*(csl-Math.sqrt(gamma*al*Math.pow(p/al,gm1/gamma)));
	    urss=ur+(p-pr)/ws(p,pr,one/rr);
	    ursr=ur-two/gm1*(csr-Math.sqrt(gamma*ar*Math.pow(p/ar,gm1/gamma)));
	    ulstar = (p > pl) ? ulss : ulsr;
	    urstar = (p > pr) ? urss : ursr;

	    //---compute inverse density -tau----------
	    taulss = one/rl - (p-pl)/Math.pow(ws(p,pl,one/rl),two);
	    taulsr = Math.pow(p/al,-one/gamma);
	    taurss = one/rr - (p-pr)/Math.pow(ws(p,pr,one/rr),two);
	    taursr = Math.pow(p/ar,-one/gamma);
	    taulstar = (p>pl) ? taulss : taulsr;
	    taurstar = (p>pr) ? taurss : taursr;

	    //----compute sound speed on left and right of contact surf
	    clstar = Math.sqrt(gamma*p*taulstar);
	    crstar = Math.sqrt(gamma*p*taurstar);

	    //----compute inverse velocity derivatives for Newton iteration
	    zlr = taulstar/clstar;
	    zls = (Math.pow(rl*csl,two)+Math.pow(ws(p,pl,one/rl),two))/
		(two*Math.pow(ws(p,pl,one/rl),3.0));
	    zrr = taurstar/crstar;
	    zrs = (Math.pow(rr*csr,two)+Math.pow(ws(p,pr,one/rr),two))/
		(two*Math.pow(ws(p,pr,one/rr),3.0));
	    zl = (p < pl) ? zlr:zls;
	    zr = (p < pr) ? zrr:zrs;

	    //----check for convergence------------------------
	    y = 1.0e-6*Math.max(Math.max(csl,csr),Math.max(Math.abs(ul),
		Math.abs(ur)));
	    z = Math.abs(ulstar-urstar);

	    if(z < y || p == pold) {
	    	break;
	    } else {
		pold = p;
		p += ((ulstar-urstar)/(zl + zr));
		p = Math.max(p,Math.max(psml,psmr));
	        ncounter++;
	    }
	}

	pstar = p;
	ustar = half*(ulstar+urstar);

	//---compute x-t space characteristics and extract r,u,p at the
	//---xbyt characteristic (zero) in x-t space to compute Flux 
	//--- l abbrev'n for lambda, i.e. characteristics

	double l0,llmr,llms,lrmr,lrms,llm,lrm,llpr,llps,lrpr,lrps,llp,lrp,c;

	l0 = ustar;

	llmr = ul - csl;
	llms = ul - ws(pstar,pl,one/rl)/rl;
	lrmr = ustar - clstar;
	lrms = llms;
	llm = (pstar < pl) ? llmr : llms;
	lrm = (pstar < pl) ? lrmr : lrms;

	llpr = ustar + crstar;
	llps = ur + ws(pstar,pr,one/rr)/rr;
	lrpr = ur + csr;
	lrps = llps;
	llp = (pstar < pr) ? llpr : llps;
	lrp = (pstar < pr) ? lrpr : lrps;

	if(xbyt <= llm) {
	    r = rl;
	    u = ul;
	    p = pl;
	} else if (xbyt > llm && xbyt <= lrm) {
	    c = (ul+two*csl/gm1-xbyt)*gm1/gp1;
	    r = Math.pow((c*c/gamma/al),(one/gm1));
	    u = xbyt + c; 
	    p = r*c*c/gamma;
	} else if (xbyt > lrm && xbyt <= l0) {
	    r = one/taulstar;
	    u = ustar;
	    p = pstar;
	} else if (xbyt > l0 && xbyt <= llp) {
	    r = one/taurstar;
	    u = ustar;
	    p = pstar;
	} else if (xbyt > llp && xbyt <= lrp) {
	    c = -(ur-two*csr/gm1-xbyt)*gm1/gp1;
	    r = Math.pow((c*c/gamma/ar),(one/gm1));
	    u = xbyt - c;
	    p = r*c*c/gamma;
	} else if (xbyt > lrp) {
	    r = rr;
	    u = ur;
	    p = pr;
	}
	computeFlux(r,u,p);

    }
	
    private double ws(double p, double p0, double tau0) {
	double sqrflux = gamma*p0/tau0*(one+gp1*half/gamma*(p-p0)/p0);
	double flux = Math.sqrt(sqrflux);
	return flux;
    }

    protected void computeFlux(double r, double u, double p) {
        massflux = r*u;
	momentumflux = p + r*u*u;
	energyflux = r*u*(gamma*p/gm1/r+half*u*u);
    }

    public double getMassFlux() {
	return massflux;
    }

    public double getMomentumFlux() {
	return momentumflux;
    }

    public double getEnergyFlux() {
	return energyflux;
    }
}
