package spec.benchmarks._239_nih; 
// Processes 16-bit signed images.

import java.util.*;
import java.awt.*;
import java.awt.image.*;

class ShortProcessor extends ImageProcessor {

	static final int MIN_SHORT = -32768;
	static final int MAX_SHORT = 32767;
	
	private int min, max, snapshotMin, snapshotMax;
	private short[] pixels;
	private short[] snapshotPixels = null;
	private boolean invert = false;
	private byte[] LUT = null;
	private short fgColor = (short)32767;
	private short bgColor = (short)0;
	private ColorModel cm;


	ShortProcessor(short[] pixels, int width, int height, ColorModel cm, ProgressBar bar) {
		this.width = width;
		this.height = height;
		this.pixels = pixels;
		this.cm = cm;
		setRoi(null);
		progressBar = bar;
		findMinAndMax();
	}


	void findMinAndMax() {
		min = 32767;
		max = -32768;
		for (int i=0; i < width*height; i++) {
			int value = pixels[i];
			if (value<min)
				min = value;
			if (value>max)
				max = value;
		}
		hideProgress();
	}


	Image createImage() {
		byte[] pixels8 = new byte[width*height];
		Image img;
				
		double scale = 255.0/(max-min);
		for (int i=0; i < width*height; i++)
			pixels8[i] = (byte)((int)(scale*(pixels[i]-min))&0xff);
	    img = Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(width, height, cm, pixels8, 0, width));
        return img;
	}
	

	Image createImage(Rectangle srcRect, int dstWidth, int dstHeight) {return null;}


	void snapshot() {
	//Make a snapshot of the current image
		snapshotWidth=width;
		snapshotHeight=height;
		snapshotMin=min;
		snapshotMax=max;
		if (snapshotPixels==null || (snapshotPixels!=null && snapshotPixels.length!=pixels.length))
			snapshotPixels = new short[width * height];
        System.arraycopy(pixels, 0, snapshotPixels, 0, width*height);
	}
	

	void reset() {
	//Reset the image from snapshot
		if (snapshotPixels!=null) {
		    width=snapshotWidth;
    		height=snapshotHeight;
		    min=snapshotMin;
    		max=snapshotMax;
    		pixels = new short[width * height];
            System.arraycopy(snapshotPixels,0,pixels,0,width*height);
    	}
	}
	

	void reset(int[] mask) {
	// Restore pixels that are within roi but not part of mask
		for (int y=roiY, my=0; y<(roiY+roiHeight); y++, my++) {
			int i = y * width + roiX;
			int mi = my * roiWidth;
			for (int x=roiX; x<(roiX+roiWidth); x++) {
				if (mask[mi++]==-1)
					pixels[i] = snapshotPixels[i];
				i++;
			}
		}
	}


	float getMin() {
		return min;
	}


	float getMax() {
		return max;
	}


	float getPixel(int x, int y) {
		if (x>=0 && x<width && y>=0 && y<height)
			return pixels[y*width + x];
		else
			return 0f;
	}


	Object getPixels() {
		return (Object)pixels;
	}


	void insert(ImageProcessor ip, int xloc, int yloc) {}
	
	
	void applyTable(byte[] lut) {}


	private void process(int op, double value) {
		double SCALE = 255.0/Math.log(255.0);
		int v1, v2;
		
		for (int y=roiY; y<(roiY+roiHeight); y++) {
			int i = y * width + roiX;
			for (int x=roiX; x<(roiX+roiWidth); x++) {
				v1 = pixels[i];
				switch(op) {
					case INVERT:
						v2 = max - (v1 - min);
						break;
					case CLEAR:
						v2 = min;
						break;
					case FILL:
						v2 = max;
						break;
					case ADD:
						v2 = v1 + (int)value;
						break;
					case MULT:
						v2 = (int)Math.round(v1 * value);
						break;
					case AND:
						v2 = v1 & (int)value;
						break;
					case OR:
						v2 = v1 | (int)value;
						break;
					case XOR:
						v2 = v1 ^ (int)value;
						break;
					case GAMMA:
						v2 = (int)(Math.exp(value*Math.log(v1/255.0))*255.0);
						break;
					case LOG:
						if (v1==0)
							v2 = 0;
						else
							v2 = (int)(Math.log(v1) * SCALE);
						break;
					 default:
					 	v2 = v1;
				}
				if (v2 < MIN_SHORT)
					v2 = MIN_SHORT;
				if (v2 > MAX_SHORT)
					v2 = MAX_SHORT;
				pixels[i++] = (short)v2;
			}
			if (y%20==0)
				showProgress((double)(y-roiY)/roiHeight);
		}
		findMinAndMax();
    }


	void invert() {process(INVERT, 0.0);}
	void clear() {process(CLEAR, 0.0);}
	void fill() {process(FILL, 0.0);}

	void add(int value) {process(ADD, value);}
	void multiply(double value) {process(MULT, value);}
	void and(int value) {process(AND, value);}
	void or(int value) {process(OR, value);}
	void xor(int value) {process(XOR, value);}
	void gamma(double value) {process(GAMMA, value);}
	void log() {process(LOG, 0.0);}


	void smooth() {
		short[] pixels2;
		int i, x, y, offset, rowOffset;
		int value;
		int p1, p2, p3, p4, p5, p6, p7, p8, p9;
	
		pixels2 = new short[width * height];
        System.arraycopy(pixels, 0, pixels2, 0, width*height);
        rowOffset = width;
		for (y = yMin; y <= yMax; y++) {
			offset = xMin + y * rowOffset + 1;
			p2 = pixels2[offset - width - 2];
			p3 = pixels2[offset - width - 1];
			p5 = pixels2[offset] - 2;
			p6 = pixels2[offset - 1];
			p8 = pixels2[offset + width - 2];
			p9 = pixels2[offset + width - 1];
			offset = xMin + y * rowOffset;
			for (x = xMin; x <= xMax; x++) {
				p1 = p2;
				p2 = p3;
				p3 = pixels2[offset - rowOffset + 1];
				p4 = p5;
				p5 = p6;
				p6 = pixels2[offset + 1];
				p7 = p8;
				p8 = p9;
				p9 = pixels2[offset + rowOffset + 1];
				value = (p1 + p2 + p3 + p4 + p5 + p6 + p7 + p8 + p9)/9;
				pixels[offset++] = (short)value;
			}
			if (y%20==0)
				showProgress((double)(y-roiY)/roiHeight);
		}
		findMinAndMax();
	}
	

	void sharpen() {
		short[] pixels2;
		int i, x, y, offset, rowOffset;
		int value;
		int p1, p2, p3, p4, p5, p6, p7, p8, p9;
	
		pixels2 = new short[width * height];
        System.arraycopy(pixels,0,pixels2,0,width*height);
        rowOffset = width;
		for (y = yMin; y <= yMax; y++) {
			offset = xMin + y * rowOffset + 1;
			p2 = pixels2[offset - width - 2];
			p3 = pixels2[offset - width - 1];
			p5 = pixels2[offset] - 2;
			p6 = pixels2[offset - 1];
			p8 = pixels2[offset + width - 2];
			p9 = pixels2[offset + width - 1];
			offset = xMin + y * rowOffset;
			for (x = xMin; x <= xMax; x++) {
				p1 = p2;
				p2 = p3;
				p3 = pixels2[offset - rowOffset + 1];
				p4 = p5;
				p5 = p6;
				p6 = pixels2[offset + 1];
				p7 = p8;
				p8 = p9;
				p9 = pixels2[offset + rowOffset + 1];
				value =  (p5*12 - p1 - p2 - p3 - p4 - p6 - p7 - p8 - p9) / 4;
				pixels[offset++] = (short)value;
			}
			if (y%20==0)
				showProgress((double)(y-roiY)/roiHeight);
		}
		findMinAndMax();
	}
	

	void findEdges() {
		short[] pixels2;
		int i, x, y, offset, rowOffset;
		int sum1, sum2, value;
		int p1, p2, p3, p4, p5, p6, p7, p8, p9;
	
		pixels2 = new short[width * height];
        System.arraycopy(pixels,0,pixels2,0,width*height);
        rowOffset = width;
		for (y = yMin; y <= yMax; y++) {
			offset = xMin + y * rowOffset + 1;
			p2 = pixels2[offset - width - 2];
			p3 = pixels2[offset - width - 1];
			p5 = pixels2[offset] - 2;
			p6 = pixels2[offset - 1];
			p8 = pixels2[offset + width - 2];
			p9 = pixels2[offset + width - 1];
			offset = xMin + y * rowOffset;
			for (x = xMin; x <= xMax; x++) {
				p1 = p2;
				p2 = p3;
				p3 = pixels2[offset - rowOffset + 1];
				p4 = p5;
				p5 = p6;
				p6 = pixels2[offset + 1];
				p7 = p8;
				p8 = p9;
				p9 = pixels2[offset + rowOffset + 1];
	        	sum1 = p1 + 2*p2 + p3 - p7 -2*p8 - p9;
	        	sum2 = p1  + 2*p4 + p7 - p3 - 2*p6 - p9;
	        	value = (int)Math.sqrt(sum1*sum1 + sum2*sum2);
				pixels[offset++] = (short)value;
			}
			if (y%20==0)
				showProgress((double)(y-roiY)/roiHeight);
		}
		findMinAndMax();
	}


    void noise(double range) {
		Random rnd=new Random();
		int v;

		for (int y=roiY; y<(roiY+roiHeight); y++) {
			int i = y * width + roiX;
			for (int x=roiX; x<(roiX+roiWidth); x++) {
				int RandomBrightness = (int)(rnd.nextGaussian()*range);
				v = pixels[i] + RandomBrightness;
				if (v < MIN_SHORT)
					v = MIN_SHORT;
				if (v > MAX_SHORT)
					v = MAX_SHORT;
				pixels[i] = (short)v;
				i++;
			}
			if (y%20==0)
				showProgress((double)(y-roiY)/roiHeight);
		}
		findMinAndMax();
    }


	public void crop() {}
	void scale(double xScale, double yScale, boolean resizeImage) {}
	void rotate(double angle) {}
	void flipVertical() {}
	void flipHorizontal() {}
	void threshold(int level) {}
	void autoThreshold() {}
	void enhanceContrast() {}
	void medianFilter() {}

}

