package spec.benchmarks._239_nih; 
// This class does image type conversions

import java.awt.*;
import java.awt.image.*;

class Converter {

	 static ColorModel makeGrayscaleColorModel(boolean invert) {
		byte[] rLUT, gLUT,bLUT;
		int k;
		
		rLUT = new byte[256];
		gLUT = new byte[256];
		bLUT = new byte[256];
		if (invert)
			for(int i=0; i<256; i++) {
				rLUT[255-i]=(byte)(i & 0xff);
				gLUT[255-i]=(byte)(i & 0xff);
				bLUT[255-i]=(byte)(i & 0xff);
			}
		else
			for(int i=0; i<256; i++) {
				rLUT[i]=(byte)(i & 0xff);
				gLUT[i]=(byte)(i & 0xff);
				bLUT[i]=(byte)(i & 0xff);
			}
		return(new IndexColorModel(8, 256, rLUT, gLUT, bLUT));
	}


	static void convert(String item, ImagePlus imp) {
		int type = imp.getType();
		boolean isColor = (type==imp.COLOR_RGB) || (type==imp.COLOR_256);
		String msg = "Converting to " + item;
		Info.showStatus(msg + "...");
	 	long start = System.currentTimeMillis();
		
		if (item.equals("8-bit Grayscale") && isColor)
			convertToGrayscale(imp);
		else if (item.equals("8-bit Color") && (type==imp.COLOR_RGB))
				convertRGBtoIndexedColor(imp);
		else if (item.equals("RGB Color")) {
			if ((type==imp.GRAY8) || (type==imp.COLOR_256))
				convertToRGB(imp);
			else if (type==imp.RGBA)
				convertRGBAtoRGB(imp);
			else if (type==imp.HSB)
				convertHSBtoRGB(imp);
			else
				unsupportedConversion(imp);
		}
		else if (item.equals("RGBA") && isColor)
				convertToRGBA(imp);
		else if (item.equals("HSB") && isColor)
				convertToHSB(imp);
		else if (item.equals("16-bit Grayscale") && (type==imp.GRAY8))
				convertToGray16(imp);
		else if (item.equals("32-bit Grayscale") && (type==imp.GRAY8))
				convertToGray32(imp);
		else if (item.equals("8-bit Grayscale") && ((type==imp.GRAY16) || (type==imp.GRAY32)))
				convertToGray8(imp);
		else {
			Info.showStatus("");
			unsupportedConversion(imp);
			Menus.updateMenus();
			return;
		}
		if (!item.equals("8-bit Color")) {
			Info.showTime(imp, start, "");
			Info.showProgress(1.0);
		}
		Menus.updateMenus();
	}
	

	static void unsupportedConversion(ImagePlus imp) {
		Info.error(
			"Supported image conversions:\n" +
			" \n" +
			"8-bit Grayscale -> 16-bit Grayscale\n" +
			"8-bit Grayscale -> 32-bit Grayscale\n" +
			"8-bit Grayscale -> RGB Color\n" +
			"16-bit Grayscale -> 8-bit Grayscale\n" +
			"32-bit Grayscale -> 8-bit Grayscale\n" +
			"8-bit Color -> RGB Color\n" +
			"RGB Color -> 8-bit Grayscale\n" +
			"RGB Color -> 8-bit Color\n" +
			"RGB Color -> RGBA (red, green, blue, alpha)\n" +
			"RGB Color -> HSB (hue, saturation, brightness)\n" +
			"RGBA -> RGB Color\n" +
			"HSB -> RGB Color\n");
	}
	
	
	static void convertToGrayscale(ImagePlus imp) {
		int c, r, g, b;
		int width, height;
		int[] pixels32;
		byte[] pixels8;
		ColorModel cm;
		Image img8;
		ImageProcessor ip;
		
		//get RGB pixels
		Info.showProgress(0.1);
		width = imp.getWidth();
		height = imp.getHeight();
		if (imp.getType()==imp.COLOR_RGB)
			ip = imp.getProcessor();
		else
			ip = new ColorProcessor(imp.getImage());
		pixels32 = (int[])ip.getPixels();
		imp.killProcessor();
		
		//convert to grayscale
		pixels8 = new byte[width * height];
		for (int i=0; i < width*height; i++) {
			c = pixels32[i];
			r = (c&0xff0000)>>16;
			g = (c&0xff00)>>8;
			b = c&0xff;
			pixels8[i] = (byte)((int)(r*0.30 + g*0.59 + b*0.11) & 0xff);
			if (i%10000==0)
				Info.showProgress(0.1 + 0.9*((double)i/(width*height)));
		}

		//create an 8-bit grayscale image
		cm = makeGrayscaleColorModel(false);
	    img8 = Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(width, height, cm, pixels8, 0, width));
	    Undo.setup(Undo.TYPE_CONVERSION, imp);
	    imp.setImage(img8);
		imp.draw();
	}


	static void convertToRGBA(ImagePlus imp) {
		
		//convert to RGBA
		ColorProcessor cp;
		Info.showProgress(0.1);
		if (imp.getType()==imp.COLOR_RGB)
			cp = (ColorProcessor)imp.getProcessor();
		else
			cp = new ColorProcessor(imp.getImage());
		int width = imp.getWidth();
		int height = imp.getHeight();
		byte[] R = new byte[width * height];
		byte[] G = new byte[width * height];
		byte[] B = new byte[width * height];
		byte[] A = new byte[width * height];
		cp.getRGBA(R, G, B, A, 0.1, 0.9);
		imp.killProcessor();
		

		//Create stack and display Red channel
		Stack stack = imp.getStack();
		stack.addSlice("Red", R);
		stack.addSlice("Green", G);
		stack.addSlice("Blue", B);
		stack.addSlice("Alpha", A);
	    Undo.reset();
		stack.setSlice(1);
	    imp.setType(imp.RGBA);
	}


	static void convertToHSB(ImagePlus imp) {

		//convert to hue, saturation and brightness
		Info.showProgress(0.1);
		ColorProcessor cp;
		if (imp.getType()==imp.COLOR_RGB)
			cp = (ColorProcessor)imp.getProcessor();
		else
			cp = new ColorProcessor(imp.getImage());
		int width = imp.getWidth();
		int height = imp.getHeight();
		byte[] H = new byte[width * height];
		byte[] S = new byte[width * height];
		byte[] B = new byte[width * height];
		cp.getHSB(H, S, B, 0.1, 0.9);
		imp.killProcessor();

		//create stack and display hue channel
		Stack stack = imp.getStack();
		stack.addSlice("Hue", H);
		stack.addSlice("Saturation", S);
		stack.addSlice("Brightness", B);
		Undo.reset();
		stack.setSlice(1);
	    imp.setType(imp.HSB);
	}
	
	
	static void convertRGBAtoRGB(ImagePlus imp) {
		Stack stack = imp.getStack();
		byte[] R = (byte[])stack.getPixels(1);
		byte[] G = (byte[])stack.getPixels(2);
		byte[] B = (byte[])stack.getPixels(3);
		byte[] A = (byte[])stack.getPixels(4);
		int width = imp.getWidth();
		int height = imp.getHeight();
		imp.killProcessor();
		ColorProcessor cp = new ColorProcessor(width, height, R, G, B, A);
		imp.setImage(cp.createImage());
		Undo.reset();
		imp.draw();
	}


	static void convertHSBtoRGB(ImagePlus imp) {
		Stack stack = imp.getStack();
		byte[] H = (byte[])stack.getPixels(1);
		byte[] S = (byte[])stack.getPixels(2);
		byte[] B = (byte[])stack.getPixels(3);
		int width = imp.getWidth();
		int height = imp.getHeight();
		imp.killProcessor();
		ColorProcessor cp = new ColorProcessor(width, height, H, S, B);
		imp.setImage(cp.createImage());
		Undo.reset();
		imp.draw();
	}
	
	
	static void convertRGBtoIndexedColor(ImagePlus imp) {
		GetNumberDialog d = new GetNumberDialog(imp.getImageJ(), imp, "Number of Colors (2-256):", 256);
	}
	
	static void convertToRGB(ImagePlus imp) {
		int width, height;
		ColorProcessor cp;

		width = imp.getWidth();
		height = imp.getHeight();
		cp = new ColorProcessor(imp.getImage());
		imp.setImage(cp.createImage());
	    Undo.setup(Undo.TYPE_CONVERSION, imp);
		imp.draw();
	}


	static void convertToGray16(ImagePlus imp) {
		int width, height;
		byte[] pixels8;
		short[] pixels16;
		ImageProcessor ip;
		
		width = imp.getWidth();
		height = imp.getHeight();
		ip = imp.getProcessor();
		pixels8 = (byte[])ip.getPixels();
		imp.killProcessor();
		pixels16 = new short[width * height];
		for (int i=0; i < width*height; i++) {
			pixels16[i] = (short)(pixels8[i]&0xff);
			if (i%10000==0)
				Info.showProgress(i/(double)(width*height));
		}
		imp.setImage(pixels16, width, height, imp.getLUT().isInverted());
		Undo.reset();
		imp.draw();
	}


	static void convertToGray32(ImagePlus imp) {
		int width, height;
		byte[] pixels8;
		float[] pixels32;
		ImageProcessor ip;
		
		width = imp.getWidth();
		height = imp.getHeight();
		ip = imp.getProcessor();
		pixels8 = (byte[])ip.getPixels();
		imp.killProcessor();
		pixels32 = new float[width * height];
		for (int i=0; i < width*height; i++) {
			pixels32[i] = (float)(pixels8[i]&0xff);
			if (i%10000==0)
				Info.showProgress(i/(double)(width*height));
		}
		imp.setImage(pixels32, width, height, imp.getLUT().isInverted());
		Undo.reset();
		imp.draw();
	}


	static void convertToGray8(ImagePlus imp) {
		ImageProcessor ip;
		Image img;
		
		ip = imp.getProcessor();
		img = ip.createImage();
		imp.killProcessor();
		imp.setImage(img);
		Undo.reset();
		imp.draw();
	}
}
