package spec.benchmarks._239_nih; 
import spec.harness.Context; 
// Processes 8-bit images. Based on the ImageProcessor class from "KickAss Java
// Programming" by Tonny Espeset (http://www.sn.no/~espeset)

import java.util.*;
import java.awt.*;
import java.awt.image.*;

class ByteProcessor extends ImageProcessor {

	private byte[] pixels;
	private byte[] snapshotPixels = null;
	private ColorModel cm;
	private boolean invert = false;
	private byte[] LUT = null;
	private byte fgColor = (byte)0;
	private byte bgColor = (byte)0xff;


	ByteProcessor(Image img) {
		width = img.getWidth(null);
		height = img.getHeight(null);
		setRoi(null);

        //Allocate buffer to hold the image's pixels
		pixels = new byte[width * height];

		//Grab pixels
		BytePixelGrabber pg = new BytePixelGrabber(img, 0, 0, width, height, pixels, 0, width);
		try {
			pg.grabPixels();
			cm = pg.getColorModel();
		}
		catch (InterruptedException e){};
	}


	ByteProcessor(Image img, ProgressBar bar) {
		this(img);
		progressBar = bar;
	}


	ByteProcessor(int width, int height, byte[] pixels) {
		this.width = width;
		this.height = height;
		setRoi(null);
		this.pixels = pixels;
	}


	Image createImage() {
	    Image img = Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(width, height, cm, pixels, 0, width));
        return img;
	}
	

	Image createImage(Rectangle srcRect, int dstWidth, int dstHeight) {
	// zoom using pixel replication
		int imag = dstWidth/srcRect.width;
		//Info.write(srcRect.width + " " + srcRect.height + " " + dstWidth + " " + dstHeight);
		
		byte[] pixels2 = new byte[dstWidth*dstHeight];
		int i = 0;
		for (int y=0; y<dstHeight; y++) {
			int j = (srcRect.y + y/imag)*width + srcRect.x;
			for (int x=0; x<dstWidth; x++)
				pixels2[i++] = pixels[j+x/imag];
		}	    
		return Toolkit.getDefaultToolkit().createImage(new MemoryImageSource(dstWidth, dstHeight, cm, pixels2, 0, dstWidth));
	}


	void snapshot() {
	//Make a snapshot of the current image
		snapshotWidth=width;
		snapshotHeight=height;
		if (snapshotPixels==null || (snapshotPixels!=null && snapshotPixels.length!=pixels.length))
		snapshotPixels = new byte[width * height];
		System.arraycopy(pixels, 0, snapshotPixels, 0, width*height);
		newSnapshot = true;
	}
	

	void reset() {
	//Reset the image from snapshot
		if (snapshotPixels!=null) {
			if (width!=snapshotWidth || height!=snapshotHeight) {
		    	width=snapshotWidth;
    			height=snapshotHeight;
    			pixels = new byte[width * height];
    		}
            System.arraycopy(snapshotPixels,0,pixels,0,width*height);
            newSnapshot = true;
    	}
	}
	

	void reset(int[] mask) {
	// Restore pixels that are within roi but not part of mask
		for (int y=roiY, my=0; y<(roiY+roiHeight); y++, my++) {
			int i = y * width + roiX;
			int mi = my * roiWidth;
			for (int x=roiX; x<(roiX+roiWidth); x++) {
				if (mask[mi++]==-1)
					pixels[i] = snapshotPixels[i];
				i++;
			}
		}
	}


	byte[] getPixelsCopy() {
		if (newSnapshot)
			return snapshotPixels;
		else {
			byte[] pixels2 = new byte[width*height];
        	System.arraycopy(pixels, 0, pixels2, 0, width*height);
			return pixels2;
		}
	}


	float getPixel(int x, int y) {
		if (x>=0 && x<width && y>=0 && y<height)
			return pixels[y*width + x] & 0xff;
		else
			return 0f;
	}


	Object getPixels() {
		return (Object)pixels;
	}


	void insert(ImageProcessor ip, int xloc, int yloc) {
	//inserts the image in ip at (xloc, yloc)
		Rectangle r1, r2;
		int srcIndex, dstIndex;
		int srcWidth, srcHeight;
		int xSrcBase, ySrcBase;
		byte[] srcPixels;
		
		srcWidth = ip.getWidth();
		srcHeight = ip.getHeight();
		r1 = new Rectangle(srcWidth, srcHeight);
		r1.move(xloc, yloc);
		r2 = new Rectangle(width, height);
		if (!r1.intersects(r2))
			return;
		srcPixels = (byte [])ip.getPixels();
		r1 = r1.intersection(r2);
		xSrcBase = (xloc<0)?-xloc:0;
		ySrcBase = (yloc<0)?-yloc:0;
		for (int y=r1.y; y<(r1.y+r1.height); y++) {
			srcIndex = (y-yloc)*srcWidth + (r1.x-xloc);
			dstIndex = y * width + r1.x;
			for (int x=r1.x; x<(r1.x+r1.width); x++)
				pixels[dstIndex++] = srcPixels[srcIndex++];
			if (y%20==0)
				showProgress((double)(y-r1.y)/r1.height);
		}
		hideProgress();
	}
	
	
	/* Filters start here */

	void applyTable(byte[] lut) {
		for (int y=roiY; y<(roiY+roiHeight); y++) {
			int i = y * width + roiX;
			for (int x=roiX; x<(roiX+roiWidth); x++) {
				pixels[i] = lut[pixels[i] & 0xff];
				i++;
			}
			if (y%20==0)
				showProgress((double)(y-roiY)/roiHeight);
		}
		hideProgress();
	}


	void smooth() {
		byte[] pixels2;
		int i, x, y, offset, rowOffset;
		int sum;
		int p1, p2, p3, p4, p5, p6, p7, p8, p9;
	
		pixels2 = getPixelsCopy();
        rowOffset = width;
		for (y = yMin; y <= yMax; y++) {
			offset = xMin + y * rowOffset + 1;
			p2 = (pixels2[offset - width - 2] & 0xff);
			p3 = (pixels2[offset - width - 1] & 0xff);
			p5 = (pixels2[offset] & 0xff - 2);
			p6 = (pixels2[offset - 1] & 0xff);
			p8 = (pixels2[offset + width - 2] & 0xff);
			p9 = (pixels2[offset + width - 1] & 0xff);
			offset = xMin + y * rowOffset;
			for (x = xMin; x <= xMax; x++) {
				p1 = p2;
				p2 = p3;
				p3 = (pixels2[offset - rowOffset + 1] & 0xff);
				p4 = p5;
				p5 = p6;
				p6 = (pixels2[offset + 1] & 0xff);
				p7 = p8;
				p8 = p9;
				p9 = (pixels2[offset + rowOffset + 1] & 0xff);
				sum = p1 + p2 + p3 + p4 + p5 + p6 + p7 + p8 + p9;
				pixels[offset++] = (byte)((sum / 9) & 0xff);
			}
			if (y%20==0)
				showProgress((double)(y-roiY)/roiHeight);
		}
		hideProgress();
	}
	

	void sharpen() {
		byte[] pixels2;
		int i, x, y, offset, rowOffset;
		int sum;
		int p1, p2, p3, p4, p5, p6, p7, p8, p9;
	
		pixels2 = getPixelsCopy();
        rowOffset = width;
		for (y = yMin; y <= yMax; y++) {
			offset = xMin + y * rowOffset + 1;
			p2 = (pixels2[offset - width - 2] & 0xff);
			p3 = (pixels2[offset - width - 1] & 0xff);
			p5 = (pixels2[offset] & 0xff - 2);
			p6 = (pixels2[offset - 1] & 0xff);
			p8 = (pixels2[offset + width - 2] & 0xff);
			p9 = (pixels2[offset + width - 1] & 0xff);
			offset = xMin + y * rowOffset;
			for (x = xMin; x <= xMax; x++) {
				p1 = p2;
				p2 = p3;
				p3 = (pixels2[offset - rowOffset + 1] & 0xff);
				p4 = p5;
				p5 = p6;
				p6 = (pixels2[offset + 1] & 0xff);
				p7 = p8;
				p8 = p9;
				p9 = (pixels2[offset + rowOffset + 1] & 0xff);
				sum =  (p5*12-p1-p2-p3-p4-p6-p7-p8-p9)/4;
				if (sum < 0)
					sum = 0;
				if (sum > 255)
					sum = 255;
				pixels[offset++] = (byte)(sum & 0xff);
			}
			if (y%20==0)
				showProgress((double)(y-roiY)/roiHeight);
		}
		hideProgress();
	}
	

	void findEdges() {
		byte[] pixels2;
		int i, x, y, offset, rowOffset;
		int sum1, sum2, sum;
		int p1, p2, p3, p4, p5, p6, p7, p8, p9;
	    long checksum = 0; 

		pixels2 = getPixelsCopy();
        rowOffset = width;
		for (y = yMin; y <= yMax; y++) {
			offset = xMin + y * rowOffset + 1;
			p2 = (pixels2[offset - width - 2] & 0xff);
			p3 = (pixels2[offset - width - 1] & 0xff);
			p5 = (pixels2[offset] & 0xff - 2);
			p6 = (pixels2[offset - 1] & 0xff);
			p8 = (pixels2[offset + width - 2] & 0xff);
			p9 = (pixels2[offset + width - 1] & 0xff);
			offset = xMin + y * rowOffset;
			for (x = xMin; x <= xMax; x++) {
				p1 = p2;
				p2 = p3;
				p3 = (pixels2[offset - rowOffset + 1] & 0xff);
				p4 = p5;
				p5 = p6;
				p6 = (pixels2[offset + 1] & 0xff);
				p7 = p8;
				p8 = p9;
				p9 = (pixels2[offset + rowOffset + 1] & 0xff);
	        	sum1 = p1 + 2*p2 + p3 - p7 -2*p8 - p9;
	        	sum2 = p1  + 2*p4 + p7 - p3 - 2*p6 - p9;
	        	sum = (int)Math.sqrt(sum1*sum1 + sum2*sum2);
	        	if (sum> 255)
	        		sum = 255;
				checksum += sum; 
				pixels[offset++] = (byte)(sum & 0xff);
			}
			if (y%20==0)
				showProgress((double)(y-roiY)/roiHeight);
		}
		spec.harness.Context.out.println("Find Edges Checksum = " + checksum%10000); 
		hideProgress();
	}


    void noise(double range) {
		Random rnd=new Random();
		int v;

		for (int y=roiY; y<(roiY+roiHeight); y++) {
			int i = y * width + roiX;
			for (int x=roiX; x<(roiX+roiWidth); x++) {
				int RandomBrightness = (int)(rnd.nextGaussian()*range);
				v = (pixels[i] & 0xff) + RandomBrightness;
				if (v < 0)
					v = 0;
				if (v > 255)
					v = 255;
				pixels[i] = (byte)(v & 0xff);
				i++;
			}
			if (y%10==0)
				showProgress((double)(y-roiY)/roiHeight);
		}
		hideProgress();
    }


	public void crop() {
		byte[] pixels2 = getPixelsCopy();
		for (int ys=roiY; ys<roiY+roiHeight; ys++) {
			int srcoff = roiX+ys*width;
			System.arraycopy(pixels2, roiX+ys*width, pixels, (ys-roiY)*roiWidth, roiWidth);
		}
		width=roiWidth;
		height=roiHeight;
		roiX = 0;
		roiY = 0;
		newSnapshot = false;
	}
	

	void scale(double xScale, double yScale, boolean resizeImage) {
		int dstWidth, dstHeight;
        int xmin, xmax, ymin, ymax;
        int srcCenterX, srcCenterY, dstCenterX, dstCenterY;

		srcCenterX = roiX + roiWidth/2;
		srcCenterY = roiY + roiHeight/2;
		if (resizeImage) {
			dstWidth = (int)(roiWidth*xScale);
			dstHeight = (int)(roiHeight*yScale);
			dstCenterX = dstWidth/2;
			dstCenterY = dstHeight/2;
	        xmin = 0;
			xmax = dstWidth - 1;
	        ymin = 0;
	        ymax = dstHeight - 1;
		}
		else {
			dstWidth = width;
			dstHeight = height;
			dstCenterX = srcCenterX;
			dstCenterY = srcCenterY;
			if ((xScale>1.0) && (yScale>1.0)) {
				//expand roi
		        xmin = dstCenterX - (int)((dstCenterX-roiX)*xScale);
		        if (xmin<0)
		        	xmin = 0;
		        xmax = xmin + (int)(roiWidth*xScale) - 1;
		        if (xmax>=width)
		        	xmax = width - 1;
		        ymin = dstCenterY - (int)((dstCenterY-roiY)*yScale);
		        if (ymin<0)
		        	ymin = 0;
		        ymax = ymin + (int)(roiHeight*yScale) - 1;
		        if (ymax>=height)
		        	ymax = height - 1;
			}
			else {
		        xmin = roiX;
				xmax = roiX + roiWidth - 1;
		        ymin = roiY;
		        ymax = roiY + roiHeight - 1;
	        }
        }
        byte[] pixels2 = getPixelsCopy();
        if (resizeImage)
        	pixels = new byte[dstWidth * dstHeight];
		boolean checkCoordinates = !resizeImage && ((xScale < 1.0) || (yScale < 1.0));
        for (int y=ymin; y<=ymax; y++) {
            int index = y*dstWidth + xmin;
			int ys = (int)((y-dstCenterY)/yScale) + srcCenterY;
            for (int x=xmin; x<=xmax; x++) {
				int xs = (int)((x-dstCenterX)/xScale) + srcCenterX;
                if (checkCoordinates && ((xs<xmin) || (xs>xmax) || (ys<ymin) || (ys>ymax)))
                	  pixels[index++] = bgColor;
                else
                	  pixels[index++] = pixels2[width*ys+xs];
            }
			if (y%20==0)
				showProgress((double)(y-ymin)/dstHeight);
        }
        width = dstWidth;
        height = dstHeight;
        if (resizeImage)
        	setRoi(null);
		hideProgress();
	}


	void rotate(double angle) {
        if (angle%360==0)
        	return;
        byte[] pixels2 = getPixelsCopy();
		int centerX = roiX + roiWidth/2;
		int centerY = roiY + roiHeight/2;
		int width = this.width;  //Local variables are faster than instance variables
        int height = this.height;
        int roiX = this.roiX;
        int xMax = roiX + this.roiWidth - 1;
        int SCALE = 1024;

        //Convert from degrees to radians and calculate cos and sin of angle
        //Negate the angle to make sure the rotation is clockwise
        double angleRadians = -angle/(180.0/Math.PI);
        int ca = (int)(Math.cos(angleRadians)*SCALE);
        int sa = (int)(Math.sin(angleRadians)*SCALE);
        int temp1 = centerY*sa - centerX*ca;
        int temp2 = -centerX*sa - centerY*ca;
        
        for (int y=roiY; y<(roiY + roiHeight); y++) {
            int index = y*width + roiX;
            int temp3 = (temp1 - y*sa)/SCALE + centerX;
            int temp4 = (temp2 + y*ca)/SCALE + centerY;
            for (int x=roiX; x<=xMax; x++) {
                //int xs = (int)((x-centerX)*ca-(y-centerY)*sa)+centerX;
                //int ys = (int)((y-centerY)*ca+(x-centerX)*sa)+centerY;
                int xs = (x*ca)/SCALE + temp3;
                int ys = (x*sa)/SCALE + temp4;
                if ((xs>=0) && (xs<width) && (ys>=0) && (ys<height))
                	  pixels[index++] = pixels2[width*ys+xs];
                else
                	  pixels[index++] = bgColor;
            }
		if (y%20==0)
			showProgress((double)(y-roiY)/roiHeight);
        }
		hideProgress();
	}


	void flipVertical() {
		byte[] pixels2 = getPixelsCopy();
		int lastLine=width*(roiY + roiHeight) + roiX - width;
		for (int offset=width*roiY+roiX; offset<(width*(roiY + roiHeight)); offset+=width) {
			System.arraycopy(pixels2, offset, pixels, lastLine, roiWidth);
			lastLine -= width;
		}
		newSnapshot = false;
	}


	void flipHorizontal() {
		byte[] pixels2 = getPixelsCopy();
        int temp = 2*roiX+roiWidth-1;
        for (int y=roiY; y<(roiY + roiHeight); y++) {
            int yOffset = y*width;
            for (int x=roiX; x<(roiX + roiWidth); x++)
				pixels[x+yOffset] = pixels2[(temp-x)+yOffset];
		}
		newSnapshot = false;
	}


	int[] getHistogram() {
		if (mask!=null)
			return getHistogram(mask);
		int[] histogram = new int[256];
		for (int y=roiY; y<(roiY+roiHeight); y++) {
			int i = y * width + roiX;
			for (int x=roiX; x<(roiX+roiWidth); x++) {
				int v = pixels[i++] & 0xff;
				histogram[v]++;
			}
		}
		return histogram;
	}


	int[] getHistogram(int[] mask) {
		int v;
		int[] histogram = new int[256];
		for (int y=roiY, my=0; y<(roiY+roiHeight); y++, my++) {
			int i = y * width + roiX;
			int mi = my * roiWidth;
			for (int x=roiX; x<(roiX+roiWidth); x++) {
				if (mask[mi++]!=-1) {
					v = pixels[i] & 0xff;
					histogram[v]++;
				}
				i++;
			}
		}
		return histogram;
	}


	void threshold(int level) {
		for (int i=0; i<width*height; i++) {
			if ((pixels[i] & 0xff) <= level)
				pixels[i] = 0;
			else
				pixels[i] = (byte)0xff;
		}
		newSnapshot = false;
	}


	void autoThreshold() {
		//Iterative thresholding technique, described originally by Ridler & Calvard in
		//"PIcture Thresholding Using an Iterative Selection Method", IEEE transactions
		// on Systems, Man and Cybernetics, August, 1978.
		
		int level;
		int[] histogram = getHistogram();
		double result;
		
		//set to zero so erased areas won't be included in the threshold
		histogram[0] = 0;
		histogram[255] = 0;
		int min = 0;
		while ((histogram[min] == 0) && (min < 255))
			min++;
		int max = 255;
		while ((histogram[max] == 0) && (max > 0))
			max--;
		if (min >= max) {
			level = 128;
			threshold(level);
			return;
		}
		int movingIndex = min;
		do {
			int tempSum1 = 0;
			int tempSum2 = 0;
			int tempSum3 = 0;
			int tempSum4 = 0;
			for (int index = min; index <= movingIndex; index++) {
				tempSum1 += index * histogram[index];
				tempSum2 += histogram[index];
			}
			for (int index = movingIndex + 1; index <= max; index++) {
				tempSum3 += index * histogram[index];
				tempSum4 += histogram[index];
			}
			
			result = (tempSum1 / tempSum2 / 2.0) + (tempSum3 / tempSum4 / 2.0);
			movingIndex += movingIndex;
		} while (((movingIndex + 1) <= result) && (movingIndex <= (max - 1)));
		
		level = (int)Math.round(result);
		//System.out.println("level: " + level);
		threshold(level);
	}
	

	void makeLUT(int min, int max) {
		if (LUT == null)
			LUT = new byte[256];
		float slope = 255 / (max - min);
		for (int i = 0; i <= 255; i++) {
			if (i <= min)
				LUT[i] = 0;
			else if (i >= max)
				LUT[i] = (byte)(255 & 0xff);
			else {
				int y = (int)((i - min) * slope);
				LUT[i] = (byte)(y & 0xff);
			}
		}
		//for (int i = 0; i <= 255; i++) {
		//	System.out.println(" " + i + ": " + (LUT[i] & 0xff));
		//}
	}
	

	private void applyLUT() {
		if (LUT == null)
			return;
		for (int i=0; i<width*height; i++) {
			pixels[i] = LUT[pixels[i] & 0xff];
		}
		newSnapshot = false;
	}


	void enhanceContrast() {
		int[] histogram = getHistogram();
		int sum = 0;
		for (int i = 0; i < 255; i++)
			sum += histogram[i];
		int threshold = sum / 5000;
		int i = -1;
		boolean found = false;
		do {
			i++;
			found = histogram[i] > threshold;
		} while (!found && (i < 255));
		int min = i;
		i = 256;
		do {
			i--;
			found = histogram[i] > threshold;
		} while (!found && (i > 0));
		int max = i;
		//System.out.println("min: " + min);
		//System.out.println("max: " + max);
		if (max > min) {
			makeLUT(min, max);
			applyLUT();
		}
	}


	private final int findMedian (int[] values) {
	//Finds the 5th largest of 9 values
		for (int i = 1; i <= 4; i++) {
			int max = 0;
			int mj = 1;
			for (int j = 1; j <= 9; j++)
				if (values[j] > max) {
					max = values[j];
					mj = j;
				}
			values[mj] = 0;
		}
		int max = 0;
		for (int j = 1; j <= 9; j++)
			if (values[j] > max)
				max = values[j];
		return max;
	}


	void medianFilter() {
		byte[] pixels2;
		int i, x, y, offset, rowOffset;
		int sum;
		int p1, p2, p3, p4, p5, p6, p7, p8, p9;

		pixels2 = getPixelsCopy();
        int[] values = new int[10];
        rowOffset = width;
		for (y = yMin; y <= yMax; y++) {
			offset = xMin + y * rowOffset + 1;
			p2 = (pixels2[offset - width - 2] & 0xff);
			p3 = (pixels2[offset - width - 1] & 0xff);
			p5 = (pixels2[offset] & 0xff - 2);
			p6 = (pixels2[offset - 1] & 0xff);
			p8 = (pixels2[offset + width - 2] & 0xff);
			p9 = (pixels2[offset + width - 1] & 0xff);
			offset = xMin + y * rowOffset;
			for (x = xMin; x <= xMax; x++) {
				p1 = p2;
				p2 = p3;
				p3 = (pixels2[offset - rowOffset + 1] & 0xff);
				p4 = p5;
				p5 = p6;
				p6 = (pixels2[offset + 1] & 0xff);
				p7 = p8;
				p8 = p9;
				p9 = (pixels2[offset + rowOffset + 1] & 0xff);
				values[1] = p1;
				values[2] = p2;
				values[3] = p3;
				values[4] = p4;
				values[5] = p5;
				values[6] = p6;
				values[7] = p7;
				values[8] = p8;
				values[9] = p9;
				pixels[offset++] = (byte)(findMedian(values) & 0xff);
			}

			if (y%20==0)
				showProgress((double)(y-roiY)/roiHeight);
		}
		hideProgress();
	}

}

